/******************************************************************************
* This module contains RF4CE NWK type and API definitions.
*
* Copyright (c) 2008, Freescale, Inc.  All rights reserved.
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale Semiconductor.
*
******************************************************************************/
#ifndef _NWK_INTERFACE_H_
#define _NWK_INTERFACE_H_


#include "EmbeddedTypes.h" 

/************************************************************************************
*************************************************************************************
* Public macros
*************************************************************************************
************************************************************************************/
/* Define the node types supported by RF4CE */
#define gNodeTypeController_c                               0
#define gNodeTypeTarget_c                                   1

/* Define the number of the entries in the node's pair table */
#ifndef gMaxPairingTableEntries_c 
  #define gMaxPairingTableEntries_c                         5
#endif

/* Define the maximum number of discovered devices returned in a block 
   of the DiscoveryCnf message */
#define aNodeDescriptorsPerBlock                            2

/* Define masks for setting the fields inside the Capabilities NIB attribute */
#define capabilitiesBitPosition_NodeType                    0
#define capabilitiesBitPosition_PowerSource                 1
#define capabilitiesBitPosition_Security                    2
#define capabilitiesBitPosition_ChannelNormalization        3
#define maskCapabilities_NodeIsTarget_c                     (1<<capabilitiesBitPosition_NodeType)
#define maskCapabilities_NodeIsMainsPowered_c               (1<<capabilitiesBitPosition_PowerSource)
#define maskCapabilities_NodeSupportsSecurity_c             (1<<capabilitiesBitPosition_Security)
#define maskCapabilities_NodeSupportsChannelNormalization_c (1<<capabilitiesBitPosition_ChannelNormalization)

/* Define sizes of lenght variables fields in the nwk frames */
#define gSizeOfSecurityKey_c                                16
#define gSizeOfVendorString_c                               7
#define gSizeOfUserString_c                                 15
#define gMaxNrOfNodeDeviceTypes_c                           3
#define gMaxNrOfNodeProfiles_c                              7


/* Return codes from the network */
#define gNWSuccess_c              0x00    /* Requested action was completed successfuly */       
#define gNWNoOrigCapacity_c       0xb0    /* No room in the originator pair table to add a new entry */
#define gNWNoRecipCapacity_c      0xb1    /* No room in the recipient pair table to add a new entry */
#define gNWDeviceIdNotPaired_c    0xb2    /* No pair information at the location specified by deviceId */
#define gNWNoResponse_c           0xb3    /* No response was received by the originator from the recipient */
#define gNWNotPermitted_c         0xb4    /* Operation not permitted */
#define gNWDuplicatePairing_c     0xb5    /* A duplicate pairing table entry was detected in a pair request command */
#define gNWFrameCounterExpired_c  0xb6    /* The frame counter has reached its maximum value */
#define gNWDiscoveryError_c       0xb7    /* Too many unique discovery request or response command frames were received than requested */
#define gNWDiscoveryTimeout_c     0xb8    /* No discovery request or response command frames were received during discovery */
#define gNWSecurityTimeout_c      0xb9    /* The security link key exchange or recovery procedure did not complete within the required time */
#define gNWSecurityFailure_c      0xba    /* A security link key was not successfully established between both ends of a pairing link */
#define gNWInvalidParam_c         0xe8    /* One of the parameters of the API function is invalid */
#define gNWUnsupportedAttribute_c 0xf4    /* NIB Attribute not supported */
#define gNWInvalidIndex_c         0xf9    /* Table index out of range */

/* Freescale specific return codes */
#define gNWDenied_c               0x80    /* Request denied by the RF4CE layer */
#define gNWNoMemory_c             0x81    /* No memory available for the RF4CE to perform requested action */
#define gNWNodeAlreadyStarted_c   0x82    /* Received when trying to start an already started RF4CE node */
#define gNWNodeNotStarted_c       0x83    /* RF4CE node should be started before executing requested action */
#define gNWNoTimers_c             0x84    /* No free timers available to perform the requested action */
#define gNWAborted_c              0x85    /* Request action was aborted */


/* Transmission options */
#define maskTxOptions_Broadcast_c                 (1<<0)
#define maskTxOptions_UseRecipLongAddress_c       (1<<1)
#define maskTxOptions_UseAck_c                    (1<<2)
#define maskTxOptions_UseSecurity_c               (1<<3)
#define maskTxOptions_UseOneChannelOnly_c         (1<<4)
#define maskTxOptions_UseChannelDesignator_c      (1<<5)
#define maskTxOptions_VendorSpecificData_c        (1<<6)

/* Reception options */
#define maskRxOptions_Broadcast_c                 (1<<0)
#define maskRxOptions_UseSecurity_c               (1<<1)
#define maskRxOptions_VendorSpecificData_c        (1<<2)


/* Invalid device identifier */
#define gInvalidDeviceId_c                        0xFF


/************************************************************************************
*************************************************************************************
* Public type definitions
*************************************************************************************
************************************************************************************/
/* Network NIB attribute identifiers */
typedef enum 
{
  gNwkNib_FirstAttribute_c = 0x60,
  
  gNwkNib_ActivePeriod_c   = 0x60,
  gNwkNib_BaseChannel_c,
  gNwkNib_DiscoveryLQIThreshold_c,
  gNwkNib_DiscoveryRepetitionInterval_c,
  gNwkNib_DutyCycle_c,
  gNwkNib_FrameCounter_c,
  gNwkNib_IndicateDiscoveryRequests_c,
  gNwkNib_InPowerSave_c,
  gNwkNib_PairingTable_c,
  gNwkNib_MaxDiscoveryRepetitions_c,
  gNwkNib_MaxFirstAttemptCSMABackoffs_c,
  gNwkNib_MaxFirstAttemptFrameRetries_c,
  gNwkNib_MaxReportedNodeDescriptors_c,
  gNwkNib_ResponseWaitTime_c,
  gNwkNib_ScanDuration_c,
  gNwkNib_UserString_c,
  /* Freescale defined NIB attributes */
  gNwkNib_StartWithNetworkInfoFromFlash_c,
  gNwkNib_AcceptDataInAutoDiscoveryMode_c,  
  gNwkNib_KeySeedPALevel_c,  
  gNwkNib_KeepPairInfoAtDuplicatePairFail_c, 
  gNwkNib_FADisabled_c,  
  gNwkNib_LastAttribute_c
} nwkNibAttribute_t;


/* Network NIB table structure */
typedef struct nwkNib_tag 
{
  uint32_t  nwkActivePeriod;
  uint8_t   nwkBaseChannel;
  uint8_t   nwkDiscoveryLQIThreshold;
  uint32_t  nwkDiscoveryRepetitionInterval;
  uint16_t  nwkDutyCycle;
  uint32_t  nwkFrameCounter;
  bool_t    nwkIndicateDiscoveryRequests;
  bool_t    nwkInPowerSave;
  uint8_t   nwkPairingTable;
  uint8_t   nwkMaxDiscoveryRepetitions;
  uint8_t   nwkMaxFirstAttemptCSMABackoffs;
  uint8_t   nwkMaxFirstAttemptFrameRetries;
  uint8_t   nwkMaxReportedNodeDescriptors;
  uint32_t  nwkResponseWaitTime;
  uint8_t   nwkScanDuration;
  uint8_t   nwkUserString[15];
  /* Freescale defined NIB attributes */
  bool_t    nwkStartWithNetworkInfoFromFlash;
  bool_t    nwkAcceptDataInAutoDiscoveryMode;  
  uint8_t   nwkKeySeedPALevel;  
  bool_t    nwkKeepPairInfoAtDuplicatePairFail;  
  bool_t    nwkFADisabled;  
} nwkNib_t;


/* Structure of the application capabilities field inside NWK frames */
typedef struct appCapabilities_tag
{ 
    uint8_t                  bUserStringSpecified   :1;
    uint8_t                  nrSupportedDeviceTypes :2;
    uint8_t                  reserved1              :1;
    uint8_t                  nrSupportedProfiles    :3;
    uint8_t                  reserved2              :1;
}appCapabilities_t;


/* Structure of information kept inside an entry in the discovery table of a RF4CE node */
typedef struct nodeDescriptor_tag
{ 
    uint8_t                  status;
    uint8_t                  recipChannel;
    uint8_t                  recipPanId[2];
    uint8_t                  recipMacAddress[8];
    uint8_t                  recipCapabilities;
    uint8_t                  recipVendorId[2];
    uint8_t                  recipVendorString[gSizeOfVendorString_c];
    appCapabilities_t        recipAppCapabilities;
    uint8_t                  recipUserString[gSizeOfUserString_c];
    uint8_t                  recipDeviceTypeList[gMaxNrOfNodeDeviceTypes_c];
    uint8_t                  recipProfilesList[gMaxNrOfNodeProfiles_c];
    uint8_t                  requestLQI; 
}nodeDescriptor_t;


/* Structure of information kept inside an entry in the pair table of a RF4CE node */
typedef struct pairTableEntry_tag
{ 
    uint8_t                  localShortAddress[2];
    uint8_t                  recipChannel;
    uint8_t                  recipMacAddress[8];
    uint8_t                  recipPanId[2];
    uint8_t                  recipShortAddress[2];
    uint8_t                  recipCapabilities;
    uint32_t                 recipFrameCounter;
    uint8_t                  securityKey[gSizeOfSecurityKey_c];
    /* Fields that do not appear in the spec but present in the FSL implementation of RF4CE */
    uint8_t                  recipUserString[gSizeOfUserString_c];
}pairTableEntry_t;


/* NodeData Database structure */
typedef struct nodeData_tag 
{
  uint8_t                     localPanId[2];
  uint8_t                     localShortAddress[2];
  pairTableEntry_t            pairTableEntry[gMaxPairingTableEntries_c];
} nodeData_t;                
      


/* Messages used for informing the application about confirms or indications from RF4CE arrived trough the NLME SAP */
typedef enum {
  /*---------------------------*/
  gNwkNlmeStartCnf_c = 0, 
  /*---------------------------*/
  gNwkNlmeAutoDiscoveryCnf_c, 
  /*---------------------------*/
  gNwkNlmeDiscoveryCnf_c,
  gNwkNlmeDiscoveryInd_c,
  /*---------------------------*/
  gNwkNlmePairCnf_c,
  gNwkNlmePairInd_c,
  /*---------------------------*/
  gNwkNlmeUnpairCnf_c,  
  gNwkNlmeUnpairInd_c,  
  /*---------------------------*/
  gNwkNlmeCommStatusInd_c,
  /*---------------------------*/
  gNwkNlmeMax_c  
}nwkNlmeToAppMsgType_t;



/* Structure of the NLME SAP messages */


/* Start service */
/*------------------------------------------------------*/
typedef struct nwkNlmeStartCnf_tag 
{
  uint8_t                 status;
}nwkNlmeStartCnf_t;


/* Auto-Discovery Service */
/*------------------------------------------------------*/
typedef struct nwkNlmeAutoDiscoveryCnf_tag 
{
  uint8_t                 status;
  uint8_t                 origMacAddress[8];   
  uint8_t                 origPanId[2];
}nwkNlmeAutoDiscoveryCnf_t;

/* Discovery Service */
/*------------------------------------------------------*/
struct nodeDescriptorBlock_tag {  
  nodeDescriptor_t                nodeDescriptorList[aNodeDescriptorsPerBlock];
  uint8_t                         nodeDescriptorCount;
  struct nodeDescriptorBlock_tag* pNext;  
};

typedef struct nodeDescriptorBlock_tag nodeDescriptorBlock_t;


typedef struct nwkNlmeDiscoveryCnf_tag 
{
  uint8_t                   status;
  uint8_t                   nrDiscoveredNodes;
  nodeDescriptorBlock_t*    pNodeDescriptorBlocks;
  
}nwkNlmeDiscoveryCnf_t;

typedef struct nwkNlmeDiscoveryInd_tag
{
  uint8_t                 status;
  uint8_t                 origMacAddress[8];   
  uint8_t                 origCapabilities; 
  uint8_t                 origVendorId[2]; 
  uint8_t*                origVendorString; 
  appCapabilities_t       origAppCapabilities;
  uint8_t*                origUserString;
  uint8_t*                origDeviceTypeList;
  uint8_t*                origProfilesList;
  uint8_t                 requestedDeviceType;  
  uint8_t                 rxLinkQuality; 
}nwkNlmeDiscoveryInd_t;


/* Pair Service */
/*------------------------------------------------------*/
typedef struct nwkNlmePairCnf_tag 
{
  uint8_t                 status;
  uint8_t                 deviceId;
  uint8_t                 recipVendorId[2];
  uint8_t*                recipVendorString;
  appCapabilities_t       recipAppCapabilities;
  uint8_t*                recipUserString;
  uint8_t*                recipDeviceTypeList;
  uint8_t*                recipProfilesList;
}nwkNlmePairCnf_t;

typedef struct nwkNlmePairInd_tag
{
  uint8_t                 status;
  uint8_t                 origPanId[2];
  uint8_t                 origMacAddress[8];   
  uint8_t                 origCapabilities; 
  uint8_t                 origVendorId[2]; 
  uint8_t*                origVendorString;
  appCapabilities_t       origAppCapabilities;
  uint8_t*                origUserString;
  uint8_t*                origDeviceTypeList;
  uint8_t*                origProfilesList;
  uint8_t                 keyExTransferCount; 
  uint8_t                 deviceId; 
}nwkNlmePairInd_t;


/* Unpairing Service */
/*------------------------------------------------------*/
typedef struct nwkNlmeUnpairCnf_tag 
{
  uint8_t                 status;
  uint8_t                 deviceId;
}nwkNlmeUnpairCnf_t;

typedef struct nwkNlmeUnpairInd_tag 
{
  uint8_t                 deviceId;
}nwkNlmeUnpairInd_t;


/* Comm Status Service */
/*------------------------------------------------------*/
typedef struct nwkNlmeCommStatusInd_tag 
{
  uint8_t                 deviceId;
  uint8_t                 targetPanId[2];
  uint8_t                 targetAddressMode;
  uint8_t                 targetAddress[8];
  uint8_t                 status;
}nwkNlmeCommStatusInd_t;


/* General structure of a message received by the application over NLME SAP */
typedef struct nwkNlmeToAppMsg_tag 
{
  nwkNlmeToAppMsgType_t           msgType;
  union {
    /*------------------------------------------------------*/
    nwkNlmeStartCnf_t             nwkNlmeStartCnf;
    /*------------------------------------------------------*/
    nwkNlmeAutoDiscoveryCnf_t     nwkNlmeAutoDiscoveryCnf;
    /*------------------------------------------------------*/
    nwkNlmeDiscoveryCnf_t         nwkNlmeDiscoveryCnf;
    nwkNlmeDiscoveryInd_t         nwkNlmeDiscoveryInd;
    /*------------------------------------------------------*/
    nwkNlmePairCnf_t              nwkNlmePairCnf;
    nwkNlmePairInd_t              nwkNlmePairInd;
    /*------------------------------------------------------*/
    nwkNlmeUnpairCnf_t            nwkNlmeUnpairCnf;
    nwkNlmeUnpairInd_t            nwkNlmeUnpairInd;
    /*------------------------------------------------------*/
    nwkNlmeCommStatusInd_t        nwkNlmeCommStatusInd;
    /*------------------------------------------------------*/
  } msgData;
}nwkNlmeToAppMsg_t;



/* Messages used for informing the application about confirms or indications from RF4CE arrived trough the NLDE SAP */
typedef enum {
  gNwkNldeDataCnf_c       = 0, 
  gNwkNldeDataInd_c, 
  /*---------------------------*/
  gNwkNldeMax_c  
}nwkNldeToAppMsgType_t;

/* Structure of the NLDE SAP messages */

/* Data Service */
/*------------------------------------------------------*/
typedef struct nwkNldeDataCnf_tag 
{
  uint8_t                 status;
  uint8_t                 deviceId;
  uint8_t                 profileId;
}nwkNldeDataCnf_t;

typedef struct nwkNldeDataInd_tag
{
  uint8_t                 deviceId;   
  uint8_t                 profileId; 
  uint8_t                 vendorId[2]; 
  uint8_t                 LQI;
  uint8_t                 rxFlags;
  uint8_t                 dataLength; 
  uint8_t*                pData; 
}nwkNldeDataInd_t;



/* General structure of a message received by the application over NLDE SAP */
typedef struct nwkNldeToAppMsg_tag 
{
  nwkNldeToAppMsgType_t    msgType;
  union 
  {
    nwkNldeDataCnf_t       nwkNldeDataCnf;
    nwkNldeDataInd_t       nwkNldeDataInd;
    /*------------------------------------------------------*/
  } msgData;
}nwkNldeToAppMsg_t;


/************************************************************************************
*************************************************************************************
* Public prototypes
*************************************************************************************
************************************************************************************/

 
/************************************************************************************
*************************************************************************************
* Public memory declarations
*************************************************************************************
************************************************************************************/

/************************************************************************************
*************************************************************************************
* Public functions
*************************************************************************************
************************************************************************************/

/************************************************************************************
* NLME_ResetRequest
*
* A call to this function will start the node. No operation implying over the air traffic
* can be executed on a node until is started, so this is the first function the user has to call prior to 
* asking for network services. The application will receive a gNwkNlmeStartCnf_c 
* message when the operation is finished.
*
* Interface assumptions:
*   None
*   
* Return value:
*   Network error code
* 
************************************************************************************/
uint8_t NLME_ResetRequest     (
                                bool_t bSetDefaultNib
                              );

/************************************************************************************
* NLME_StartRequest
*
* A call to this function will start the node. No Discovery, Pair, Unpair or Data operation 
* can be executed on a node until is started, so this is the first function the user has 
* to call prior to asking for the above enumerated network services. The application will 
* receive a gNwkNlmeStartCnf_c message when the operation completes.
*
* Interface assumptions:
*   None
*   
* Return value:
*   Network error code
* 
************************************************************************************/
uint8_t NLME_StartRequest     (
                                void
                              );
                              
                              
/************************************************************************************
* NLME_AutoDiscoveryRequest
*
* Will enter in Auto discovery mode. For more details about the AutoDiscovery
* Request service please see the RF4CE spec document. The application will receive a 
* gNwkNlmeAutoDiscoveryCnf_c message when the operation completes.
*
* Interface assumptions:
*   None
*   
* Return value:
*   Network error code
* 
************************************************************************************/
uint8_t NLME_AutoDiscoveryRequest (

                                appCapabilities_t recipAppCapabilities,
                                uint8_t*          recipDeviceTypeList,
                                uint8_t*          recipProfileIdList,
                                uint32_t          autoDiscDuration
                              );                              
   
/************************************************************************************
* NLME_DiscoveryRequest
*
* Will determine the start of a discovery process. For more details about the Discovery
* Request service please see the RF4CE spec document. The application will receive a 
* gNwkNlmeDiscoveryCnf_c message when the operation completes.
*
* Interface assumptions:
*   None
*   
* Return value:
*   Network error code
* 
************************************************************************************/
uint8_t NLME_DiscoveryRequest (
                                uint8_t*          recipPanId, 
                                uint8_t*          recipShortAddress,
                                uint8_t           recipDeviceType,
                                appCapabilities_t origAppCapabilities,
                                uint8_t*          origDeviceTypeList,
                                uint8_t*          origProfileIdList,
                                uint8_t           discProfileIdListSize,
                                uint8_t*          discProfileIdList,
                                uint32_t          discDuration
                              );

/************************************************************************************
* NLME_DiscoveryResponse
*
* Will determine the sending of a Discovery Response frame to a RF4CE device identified
* by the values of the API parameters. For more details about the Discovery Response service 
* please see the RF4CE spec document. The application will receive a gNwkNlmeCommStatusInd_c 
* message when the operation completes.
*
* Interface assumptions:
*   None
*   
* Return value:
*   Network error code
* 
************************************************************************************/
uint8_t NLME_DiscoveryResponse(
                                uint8_t           status,
                                uint8_t*          recipMacAddress,
                                appCapabilities_t origAppCapabilities,
                                uint8_t*          origDeviceTypeList,
                                uint8_t*          origProfileIdList,
                                uint8_t           discoveryReqLQI
                              );

/************************************************************************************
* NLME_PairRequest
*
* Will determine the start of a pair process. For more details about the Pair
* Request service please see the RF4CE spec document. The application will receive a 
* gNwkNlmePairCnf_c message when the operation completes.
*
* Interface assumptions:
*   None
*   
* Return value:
*   Network error code
* 
************************************************************************************/
uint8_t NLME_PairRequest      (
                                uint8_t           recipChannel,
                                uint8_t*          recipPanId, 
                                uint8_t*          recipMacAddress,
                                appCapabilities_t origAppCapabilities,
                                uint8_t*          origDeviceTypeList,
                                uint8_t*          origProfileIdList,
                                uint8_t           keyExTransferCount
                              );

/************************************************************************************
* NLME_PairResponse
*
* Will determine the sending of a Pair Response frame to a RF4CE device identified
* by the values of the API parameters. For more details about the Pair Response service 
* please see the RF4CE spec document. The application will receive a gNwkNlmeCommStatusInd_c 
* message when the operation completes.
*
* Interface assumptions:
*   None
*   
* Return value:
*   Network error code
* 
************************************************************************************/
uint8_t NLME_PairResponse     (
                                uint8_t           status,
                                uint8_t*          recipPanId, 
                                uint8_t*          recipMacAddress,
                                appCapabilities_t origAppCapabilities,
                                uint8_t*          origDeviceTypeList,
                                uint8_t*          origProfileIdList,
                                uint8_t           deviceId
                              );

/************************************************************************************
* NLME_UnpairRequest
*
* Will determine the start of an unpair process. For more details about the Unpair
* Request service please see the RF4CE spec document. The application will receive a 
* gNwkNlmeUnpairCnf_c message when the operation completes.
*
* Interface assumptions:
*   None
*   
* Return value:
*   Network error code
* 
************************************************************************************/
uint8_t NLME_UnpairRequest    (
                                uint8_t deviceId
                              );

/************************************************************************************
* NLME_UnpairResponse
*
* Will determine the deletion of the pair information from the entry identified by the 
* deviceId index in the pair table, if an Unpair Request was previously received from the 
* device in that entry. For more details about the Unpair Response service please see the 
* RF4CE spec document. The application will receives the status of the execution of this
* request in the value returned by the API. This service is synchronous. 
*
* Interface assumptions:
*   None
*   
* Return value:
*   Network error code
* 
************************************************************************************/
uint8_t NLME_UnpairResponse   (
                                uint8_t deviceId
                              );

/************************************************************************************
* NLME_GetRequest
*
* Gets the value of a NIB attribute. For more details about the NLME_GetRequest service 
* please see the RF4CE spec document.This service is synchronous. 
*
* Interface assumptions:
*   None
*   
* Return value:
*   Network error code
* 
************************************************************************************/
uint8_t NLME_GetRequest       (
                                uint8_t  nibAttribute,
                                uint8_t  nibAttributeIndex,
                                uint8_t* nibAttributeValue
                              );

/************************************************************************************
* NLME_SetRequest
*
* Sets the value of a NIB attribute. For more details about the NLME_SetRequest service 
* please see the RF4CE spec document.This service is synchronous. 
*
* Interface assumptions:
*   None
*   
* Return value:
*   Network error code
* 
************************************************************************************/
uint8_t NLME_SetRequest       (
                                uint8_t  nibAttribute,
                                uint8_t  nibAttributeIndex,
                                uint8_t* nibAttributeValue
                              );                                                            

/************************************************************************************
* NLME_RxEnableRequest
*
* Turns the receiver on or off, enables or disables the intermitent Rx mechanism. 
* For more details about the NLME_RxEnableRequest service please see the RF4CE spec document.
* This service is synchronous. 
*
* Interface assumptions:
*   None
*   
* Return value:
*   Network error code
* 
************************************************************************************/
uint8_t NLME_RxEnableRequest(
                                uint32_t rxOnDuration
                             );

/************************************************************************************
* NLME_UpdateKeyRequest
*
* Updates the security key for a paired device. 
* For more details about the NLME_UpdateKeyRequest service please see the RF4CE spec document.
* This service is synchronous. 
*
* Interface assumptions:
*   None
*   
* Return value:
*   Network error code
* 
************************************************************************************/
uint8_t NLME_UpdateKeyRequest(
                                uint8_t   deviceId,
                                uint8_t*  newLinkKey 
                             );

/************************************************************************************
* NLDE_DataRequest
*
* Will determine the start of a data process. For more details about the Data Request 
* service please see the RF4CE spec document. The application will receive a 
* gNwkNldeDataCnf_c message when the operation completes.
*
* Interface assumptions:
*   None
*   
* Return value:
*   Network error code
* 
************************************************************************************/
uint8_t NLDE_DataRequest(
                          uint8_t   deviceId, 
                          uint8_t   profileId,
                          uint8_t*  vendorId,
                          uint8_t   nsduLength,
                          uint8_t*  nsdu,
                          uint8_t   txOptions
                        );

/************************************************************************************
* NWK_IsIdle
* 
* Function to be used by the application to determine if RF4CE layer state is idle. Returns
* TRUE when the RF4CE state is idle and FALSE otherwise. 
* It should be called before entering into low power mode or before saving data to NVM.
*
* Interface assumptions:
*   None
*   
* Return value:
*   TRUE
*   FALSE
* 
************************************************************************************/
bool_t  NWK_IsIdle(void); 


/************************************************************************************
* NWK_GetAllowedLowPowerInterval
* 
* Function to be used by the application to determine if RF4CE layer can safely enter
* a low power mode. Returns the amount of time in MAC symbols that the application 
* should enter the low power mode.
*
* Interface assumptions:
*   None
*   
* Return value:
*   TRUE
*   FALSE
* 
************************************************************************************/
uint32_t NWK_GetAllowedLowPowerInterval(void);


/************************************************************************************
* NWK_GetNodePanId
*
* Macro to get the value of the PanId of the local RF4CE node. Will return a pointer to 
* a 2 byte array containing the PanId value. 
*
* Interface assumptions:
*   None
*   
* Return value:
*   Pointer to a 2 byte array
* 
************************************************************************************/
#define NWK_GetNodePanId()        nodeData.localPanId

/************************************************************************************
* NWK_GetNodeShortAddress
*
* Macro to get the value of the ShortAddress of the local RF4CE node. Will return a pointer to 
* a 2 byte array containing the ShortAddress value. 
*
* Interface assumptions:
*   None
*   
* Return value:
*   Pointer to a 2 byte array
* 
************************************************************************************/
#define NWK_GetNodeShortAddress() nodeData.localShortAddress 

/************************************************************************************
* NWK_GenerateShortAddress
*
* Function to generate a random short address that is different from the short addresses
* of all devices in the pair table. The new short address will be generated in a 2 bytes
* memory location that should be provided by the application 
*
* Interface assumptions:
*   None
*   
* Return value:
*   None
* 
************************************************************************************/
void NWK_GenerateShortAddress (
                                uint8_t* shortAddress
                              );

/************************************************************************************
* NWK_GenerateSecurityKey
*
* Function to generate a random security key that is different from the security keys
* of all devices in the pair table. The new security key will be generated in a 16 bytes
* memory location that should be provided by the application 
*
* Interface assumptions:
*   None
*   
* Return value:
*   None
* 
************************************************************************************/
void NWK_GenerateSecurityKey  (
                                uint8_t* securityKey
                              );

/************************************************************************************
* NWK_AddNewPairTableEntry
*
* Function to add a new entry in the pair table of the local RF4CE node. Will return the 
* entry index of the new entry if the operation completes successfully, the gNWInvalidParam_c
* status if the recipChannel is not in the [15,20,25] list and the gNWNoCapacity_c status
* if there is no room to add a new entry in the pair table
*
* Interface assumptions:
*   None
*   
* Return value:
*   gNWInvalidParam_c
*   gNWNoCapacity_c
*   index of the new added entry
* 
************************************************************************************/
uint8_t NWK_AddNewPairTableEntry(
                                  uint8_t*            localShortAddress,    
                                  uint8_t             recipChannel,
                                  uint8_t*            recipMacAddress,
                                  uint8_t*            recipPanId,
                                  uint8_t*            recipShortAddress,
                                  uint8_t             recipCapabilities,
                                  uint8_t*            securityKey,
                                  uint8_t*            recipUserString
                                  );


/************************************************************************************
* NWK_SavePersistentData
*
* Function to trigger the saving of the RF4CE nwk sensitive data into Flash. This includes
* the RF4CE NIB table and RF4CE pairing table.
*
* Interface assumptions:
*   None
*   
* Return value:
*   None
* 
************************************************************************************/
void NWK_SavePersistentData   (
                                void
                              );


/************************************************************************************
* NWK_SaveFrameCounter
*
* Function to trigger the saving of the node's frame counter in Flash
*
* Interface assumptions:
*   None
*   
* Return value:
*   gNWDenied_c
*   gNWSuccess_c
* 
************************************************************************************/
uint8_t NWK_SaveFrameCounter(
                              void
                            );


/************************************************************************************
* NWK_SetMacAddress
*
* Function to set the IEEE 802.15.4 MAC address of the local node
* *
* Interface assumptions:
*   None
*   
* Return value:
*   gNWInvalidParam_c - When the pointer where to get the MAC address from is NULL
*   gNWSuccess_c
* 
************************************************************************************/
uint8_t NWK_SetMacAddress(uint8_t* pMacAddress);


/************************************************************************************
* NWK_GetMacAddress
*
* Function to get the IEEE 802.15.4 MAC address of the local node
* *
* Interface assumptions:
*   None
*   
* Return value:
*   gNWInvalidParam_c - When the pointer where to copy the MAC address to is NULL
*   gNWSuccess_c
* 
************************************************************************************/
uint8_t NWK_GetMacAddress(uint8_t* pMacAddress);


/************************************************************************************
* NWK_GetLastPacketLQI
*
* Function to get the LQI of the networks' last received packe
* *
* Interface assumptions:
*   None
*   
* Return value:
*   LQI value of the last received packet
* 
************************************************************************************/
uint8_t NWK_GetLastPacketLQI(void);


/************************************************************************************
* NWK_NLME_SapHandler
*
* Function that should be implemented in the app layer which is called by the RF4CE layer 
* in order to send NLME messages to the application. The function's code should queue 
* the received message and send an event to the application task to notify it about the 
* arrival.
*
* Interface assumptions:
*   None
*   
* Return value:
*   None
* 
************************************************************************************/
void    NWK_NLME_SapHandler   (
                                nwkNlmeToAppMsg_t* nwkNlmeToAppMsg
                              );

/************************************************************************************
* NWK_NLDE_SapHandler
*
* Function that should be implemented in the app layer which is called by the RF4CE layer 
* in order to send NLDE messages to the application. The function's code should queue 
* the received message and send an event to the application task to notify it about the 
* arrival.
*
* Interface assumptions:
*   None
*   
* Return value:
*   None
* 
************************************************************************************/
void    NWK_NLDE_SapHandler   (
                                nwkNldeToAppMsg_t* nwkNldeToAppMsg
                              );

#endif  /* _NWK_INTERFACE_H_ */
